import { NextRequest, NextResponse } from "next/server"
import { neon } from "@neondatabase/serverless"

const sql = neon(process.env.DATABASE_URL!)

// PUT - Atualizar categoria específica
export async function PUT(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    const { name } = await request.json()
    const id = params.id

    if (!name) {
      return NextResponse.json(
        { error: "Nome da categoria é obrigatório" },
        { status: 400 }
      )
    }

    const result = await sql`
      UPDATE music_categories 
      SET name = ${name}
      WHERE id = ${id}
      RETURNING id, name, created_at
    `

    if (result.length === 0) {
      return NextResponse.json(
        { error: "Categoria não encontrada" },
        { status: 404 }
      )
    }

    return NextResponse.json(result[0])
  } catch (error: any) {
    console.error("Erro ao atualizar categoria:", error)
    
    if (error.code === "23505") { // Unique constraint violation
      return NextResponse.json(
        { error: "Categoria com este nome já existe" },
        { status: 409 }
      )
    }

    return NextResponse.json(
      { error: "Erro interno do servidor" },
      { status: 500 }
    )
  }
}

// DELETE - Deletar categoria específica
export async function DELETE(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    const id = params.id

    // Verificar se há músicas usando esta categoria
    const songsCount = await sql`
      SELECT COUNT(*) as count FROM songs WHERE category_id = ${id}
    `

    if (songsCount[0].count > 0) {
      return NextResponse.json(
        { error: "Não é possível deletar categoria que possui músicas associadas" },
        { status: 409 }
      )
    }

    const result = await sql`
      DELETE FROM music_categories 
      WHERE id = ${id}
      RETURNING id, name
    `

    if (result.length === 0) {
      return NextResponse.json(
        { error: "Categoria não encontrada" },
        { status: 404 }
      )
    }

    return NextResponse.json({ message: "Categoria deletada com sucesso" })
  } catch (error) {
    console.error("Erro ao deletar categoria:", error)
    return NextResponse.json(
      { error: "Erro interno do servidor" },
      { status: 500 }
    )
  }
}